// import AddAssetHtmlPlugin from 'add-asset-html-webpack-plugin'
import chalk from 'chalk'
import CopyWebpackPlugin from 'copy-webpack-plugin'
import dotenv from 'dotenv'
import ESLintPlugin from 'eslint-webpack-plugin'
import HtmlWebpackPlugin from 'html-webpack-plugin'
import MiniCssExtractPlugin from 'mini-css-extract-plugin'
import path from 'path'
import ProgressBarPlugin from 'progress-bar-webpack-plugin'
import webpack from 'webpack'

// import AntdDayjsWebpackPlugin from 'antd-dayjs-webpack-plugin'

dotenv.config()

const Resolve = (...args: string[]) => {
  return path.resolve(__dirname, '../', ...args)
}
const isProd = process.env.NODE_ENV === 'production'

const exclude = /node_modules/
const include = /src/

const BaseConfig: webpack.Configuration = {
  entry: ['./src/main.tsx'],
  output: {
    path: Resolve('dist'),
    filename: 'js/[name].[contenthash].js',
    chunkFilename: 'js/[name].chunk.js',
    publicPath: process.env.PUBLIC_PATH
  },
  module: {
    rules: [
      {
        // 同时认识ts jsx js tsx 文件
        test: /\.(t|j)sx?$/,
        exclude,
        include,
        use: [
          {
            loader: 'babel-loader',
            options: {
              // 缓存：第二次构建时，会读取之前的缓存
              cacheDirectory: true
            }
          }
        ]
      },
      {
        test: /\.less$/,
        include: [/src/, /node_modules/],
        use: [
          isProd ? MiniCssExtractPlugin.loader : 'style-loader',
          {
            loader: 'css-loader',
            options: {
              importLoaders: 1
            }
          },
          {
            loader: 'postcss-loader',
            options: {
              postcssOptions: {
                plugins: [
                  'postcss-preset-env',
                  'autoprefixer'
                ]
              },
              sourceMap: true
            }
          },
          {
            loader: 'less-loader',
            options: {
              lessOptions: { // 如果使用less-loader@5，请移除 lessOptions 这一级直接配置选项。
                modifyVars: {
                  // 'primary-color': '#1DA57A',
                  // 'link-color': '#1DA57A',
                  // 'border-radius-base': '2px'
                },
                javascriptEnabled: true
              }
            }
          }
        ]
      },
      {
        test: /\.css$/,
        include: [/src/, /node_modules\/@ant-design/, /node_modules\/antd/, /node_modules\/@wangeditor/],
        use: [
          {
            loader: 'style-loader'
          },
          {
            loader: 'css-loader'
          }
        ]
      },
      {
        test: /\.(png|svg|jpg|gif)$/,
        exclude,
        include,
        use: [
          {
            loader: 'file-loader',
            options: {
              limit: 5000,
              esModule: false,
              // 分离图片至imgs文件夹
              name () {
                if (process.env.NODE_ENV === 'development') {
                  return '[path][name].[ext]'
                }
                return 'images/[contenthash].[ext]'
              }
            }
          }
        ]
      },
      {
        test: /\.(woff|woff2|eot|ttf|otf)$/,
        exclude,
        include,
        use: [
          {
            loader: 'file-loader',
            options: {
              name () {
                if (process.env.NODE_ENV === 'development') {
                  return '[path][name].[ext]'
                }
                return 'fonts/[contenthash].[ext]'
              }
            }
          }
        ]
      }
    ]
  },
  plugins: [
    /**
     * @description 定义全局变量插件
     */
    new webpack.EnvironmentPlugin(['NODE_ENV', 'API_HOST', 'COS_HOST']),
    /**
     * @description 复制资源插件
     */
    new CopyWebpackPlugin({
      patterns: [
        {
          from: Resolve('public'),
          to: Resolve('dist')
        }
      ]
    }),
    /**
     * @description webpack 构建进度条
     */
    new (ProgressBarPlugin as any)({
      width: 50, // 默认20，进度格子数量即每个代表进度数，如果是20，那么一格就是5。
      format: chalk.blue.bold('build') + chalk.yellow('[:bar] ') + chalk.green.bold(':percent') + ' (:elapsed秒)',
      complete: '-', // 默认“=”，完成字符
      clear: false // 默认true，完成时清除栏的选项
    }),
    new ESLintPlugin({
      fix: true /* 自动帮助修复 */,
      extensions: ['js', 'jsx', 'ts', 'tsx', 'json', 'react'],
      exclude: 'node_modules'
    }),
    new HtmlWebpackPlugin({
      inject: true,
      template: path.resolve(__dirname, '../src/index.html')
    })
    // new AntdDayjsWebpackPlugin()
    // ==== dll 配置 ====
    // new webpack.DllReferencePlugin({
    //   context: __dirname,
    //   manifest: require('../dll/vendor.manifest.json')
    // }),
    // // 打包后的 .dll.js 文件需要引入到 html中，可以通过 add-asset-html-webpack-plugin 插件自动引入
    // new AddAssetHtmlPlugin({
    //   filepath: path.resolve('./dll/vendor.dll.js'),
    //   publicPath: ''
    // })
  ],
  resolve: {
    extensions: ['.js', '.json', '.jsx', '.less', '.css', '.vue', '.json', '.ts', '.tsx'],
    alias: {
      '@': Resolve('src')
    }
  },
  cache: {
    type: 'filesystem'
  },
  watchOptions: {
    ignored: /node_modules/
  },
  /**
   * @description 设置webpack 构建时候信息输出
   */
  stats: {
    colors: true,
    modules: false,
    children: false,
    chunks: false,
    chunkModules: false
  }
}
export default BaseConfig
