import React, { useCallback, useState } from 'react'

import { Button, message, Popconfirm, Space } from 'antd'

import { DList } from '@/components/DList'
import { DTable, usePTable } from '@/components/DTable'
import { useAppSelector } from '@/hooks/useStore'

import { DelApi, GetListByPageApi } from './common/services'
import { TState } from './common/typing'
import Add from './components/add'
import UpdatePassword from './components/update-password'

const AccountList = () => {
  const { permissionsData, account } = useAppSelector(({ menu }) => (menu))
  const [dataSource, setDataSource] = useState<TState[]>([])
  const [currentId, setCurrentId] = useState<number>(NaN)
  const [isModalVisible, setIsModalVisible] = useState(false)
  const [passwordVisible, setPasswordVisible] = useState(false)
  const isSearch: any = permissionsData.includes(account?.info1) || permissionsData.includes(account?.info2)
  const { tableProps, actionRef } = usePTable<TState>({
    headerTitle: '账号列表',
    search: isSearch,
    request: useCallback(async (params: any) => {
      const result = { data: [], total: 1, success: true }
      try {
        const { pageSize: limit, current: page, username } = params
        const { data } = await GetListByPageApi({
          limit, // 每页显示记录数
          page, // 当前页码，从1开始
          username,
          order: '', // 排序方式，可选值(asc、desc)
          orderField: '' // 排序字段
        })
        const { list, total } = data
        setDataSource(list)
        result.data = list
        result.total = total
      } catch (error) {
        console.error('请求列表失败', error)
      }
      return result
    }, []),
    columns: [
      {
        title: 'ID',
        dataIndex: 'id',
        width: 100,
        copyable: true,
        hideInSearch: true
      },
      {
        title: '账号名',
        dataIndex: 'username',
        copyable: true,
        ellipsis: true,
        hideInSearch: false
      },
      {
        title: '昵称',
        key: 'realName',
        dataIndex: 'realName',
        hideInSearch: true
      },
      {
        title: '角色',
        key: 'roleIdList',
        dataIndex: 'roleIdList',
        hideInSearch: true
      },
      {
        title: '状态',
        key: 'status',
        dataIndex: 'status',
        hideInSearch: true
      },
      {
        title: '操作',
        valueType: 'option',
        render: (text, record) => [
          <Space key="operate" size={20}>
            {permissionsData.includes(account?.update) && (
              <a
                key="editable"
                onClick={() => handleClickOperation('Edit', record)}>
                编辑
              </a>
            )}
            <a
              key="password"
              onClick={() => handleClickOperation('Password', record)}>
              修改密码
            </a>
            {permissionsData.includes(account?.delete) && (
              <Popconfirm
                title="确定删除该条数据？"
                onConfirm={() => handleClickOperation('Del', record)}
                okText="Yes"
                cancelText="No"
              >
                <a key="delete"> 删除 </a>
              </Popconfirm>

            )}

          </Space>
        ]
      }
    ],
    toolBarRender: useCallback(() => [
      // 工具栏设置
      (permissionsData.includes(account?.save) && (
        <Button key="tool" type="primary" onClick={() => handleUpdateVisible(true, '')}>
          创建账号
        </Button>
      ))
    ], [])
  })

  // 表格-操作
  const handleClickOperation = useCallback(
    async (type: 'Edit' | 'Del' | 'Password', record: TState) => {
      switch (type) {
        case 'Edit':
          setIsModalVisible(true)
          setCurrentId(record.id)
          break
        case 'Password':
          setPasswordVisible(true)
          setCurrentId(record.id)
          break
        case 'Del':
          try {
            await DelApi([record.id])
            message.success('删除成功')
            dataSource.length <= 1 ? actionRef.current?.reloadAndRest?.() : actionRef.current?.reload?.()
          } catch (error) {
            console.log(error)
            message.error('删除失败')
          }
          break
      }
    },
    [dataSource]
  )
  const handleUpdateVisible = (value: boolean, type: string) => {
    setIsModalVisible(value)
    if (!value) {
      setCurrentId(NaN)
    }
    if (type === 'add') {
      (actionRef.current as any).reload()
    }
  }
  const handleUpdatePassword = (value: boolean) => {
    setPasswordVisible(value)
  }

  return (
    <>
      <DList>
        <DTable {...tableProps} />
      </DList>
      <Add visible={isModalVisible} handleUpdateVisible={handleUpdateVisible} id={currentId} />
      <UpdatePassword
        id={currentId}
        handleUpdatePassword={handleUpdatePassword}
        visible={passwordVisible}
      />
    </>
  )
}

export default AccountList
